"use server";

import { revalidatePath } from "next/cache";
import { prisma } from "@/lib/prisma";

// --- Companies ---

export async function getCompanies() {
    return await prisma.company.findMany({
        orderBy: { name: "asc" },
    });
}

export async function createCompany(name: string, template: string) {
    const slug = name.toLowerCase().replace(/ /g, "-").replace(/[^\w-]+/g, "");

    await prisma.company.create({
        data: {
            name,
            template,
            slug,
        },
    });
    revalidatePath("/");
}

export async function deleteCompany(id: string) {
    await prisma.company.delete({ where: { id } });
    revalidatePath("/");
}

// --- Posts ---

export async function getPosts(companyId: string) {
    return await prisma.post.findMany({
        where: { companyId },
        orderBy: { scheduledDate: "desc" }, // Edited to sort by scheduledDate
        take: 50,
    });
}

export async function getAllPosts() {
    return await prisma.post.findMany({
        include: { company: true },
        orderBy: { scheduledDate: "asc" },
    });
}

export async function createPost(
    companyId: string,
    content: string,
    fullText: string,
    subject: string | undefined,
    type: string,
    scheduledDate: Date
) {
    await prisma.post.create({
        data: {
            companyId,
            content,
            fullText,
            subject,
            type,
            scheduledDate,
        },
    });
    revalidatePath("/");
}

// --- Update Actions ---

export async function updateCompany(id: string, name: string, template: string) {
    // Re-generate slug if name changes? Maybe keep it simple and only update display fields.
    // Let's allow updating name and template.
    await prisma.company.update({
        where: { id },
        data: { name, template },
    });
    revalidatePath("/");
}

export async function updatePost(
    id: string,
    content: string,
    fullText: string,
    subject: string | undefined,
    type: string,
    scheduledDate: Date
) {
    await prisma.post.update({
        where: { id },
        data: {
            content,
            fullText,
            subject,
            type,
            scheduledDate,
        },
    });
    revalidatePath("/");
}

export async function deletePost(id: string) {
    await prisma.post.delete({ where: { id } });
    revalidatePath("/");
}

// --- AI ---

import { GoogleGenerativeAI } from "@google/generative-ai";

export async function generateCaption(prompt: string) {
    if (!process.env.GEMINI_API_KEY) {
        return "Hata: GEMINI_API_KEY bulunamadı. Lütfen .env dosyasına ekleyin.";
    }

    try {
        const genAI = new GoogleGenerativeAI(process.env.GEMINI_API_KEY);
        const model = genAI.getGenerativeModel({ model: "gemini-1.5-flash" });

        const result = await model.generateContent(`
            Aşağıdaki konu hakkında Türkçe, ilgi çekici, samimi ve profesyonel bir sosyal medya (Instagram) metni yaz.
            Kurallar:
            1. Hashtag (#) kesinlikle kullanma.
            2. Emojileri bol, yerinde ve doğal kullan.
            3. Çok uzun olmasın, okunabilir paragraflar olsun.
            4. Satış veya etkileşim odaklı bitir (CTA).
            
            Konu: ${prompt}
        `);

        return result.response.text();
    } catch (error) {
        console.error("AI Error:", error);
        return "Üzgünüm, AI şu anda yanıt veremiyor. Lütfen tekrar deneyin.";
    }
}
