"use client";

import { useState } from "react";
import { generateCaption } from "@/app/actions";

interface AIModalProps {
    onClose: () => void;
    onGenerate: (text: string) => void;
}

export default function AIModal({ onClose, onGenerate }: AIModalProps) {
    const [prompt, setPrompt] = useState("");
    const [isLoading, setIsLoading] = useState(false);

    const handleGenerate = async () => {
        if (!prompt.trim()) return;
        setIsLoading(true);
        try {
            const result = await generateCaption(prompt);
            onGenerate(result);
            onClose();
        } catch (error) {
            console.error("AI Generation failed", error);
            // Optionally show an error toast here via prop or context
        } finally {
            setIsLoading(false);
        }
    };

    return (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/50 backdrop-blur-sm p-4">
            <div className="w-full max-w-lg bg-white rounded-xl shadow-xl p-6 relative animate-in zoom-in-95 duration-200">
                <button
                    onClick={onClose}
                    className="absolute top-4 right-4 text-zinc-400 hover:text-zinc-600"
                >
                    ✕
                </button>
                <div className="flex items-center gap-2 mb-4">
                    <div className="p-2 bg-purple-100 text-purple-600 rounded-lg">
                        <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M13 10V3L4 14h7v7l9-11h-7z" /></svg>
                    </div>
                    <h2 className="text-xl font-bold text-zinc-900">AI Yazarı</h2>
                </div>

                <p className="text-sm text-zinc-500 mb-4">
                    İstediğiniz içeriği kısaca tarif edin, yapay zeka sizin için ilgi çekici bir metin oluştursun.
                </p>

                <div className="space-y-4">
                    <textarea
                        rows={4}
                        className="w-full rounded-md border-zinc-300 shadow-sm focus:border-purple-500 focus:ring-purple-500 py-3 px-4 text-sm resize-none"
                        placeholder="Örn: Yaz sezonu ürünlerimiz için esprili ve heyecan verici bir Instagram postu hazırla..."
                        value={prompt}
                        onChange={(e) => setPrompt(e.target.value)}
                        autoFocus
                    />

                    <button
                        onClick={handleGenerate}
                        disabled={isLoading || !prompt.trim()}
                        className="w-full bg-purple-600 hover:bg-purple-700 text-white font-semibold py-3 px-4 rounded-md transition-colors disabled:opacity-50 flex items-center justify-center gap-2"
                    >
                        {isLoading ? (
                            <>
                                <svg className="animate-spin h-5 w-5 text-white" fill="none" viewBox="0 0 24 24"><circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle><path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path></svg>
                                Oluşturuluyor...
                            </>
                        ) : (
                            <>
                                ✨ Oluştur
                            </>
                        )}
                    </button>
                </div>
            </div>
        </div>
    );
}
