"use client";

import { useState } from "react";
import { createPost, deletePost, deleteCompany, updatePost } from "@/app/actions";
import { useRouter } from "next/navigation";
import type { Company, Post } from "@prisma/client";
import Toast, { ToastType } from "./Toast";
import EditCompanyModal from "./EditCompanyModal";
import AIModal from "./AIModal";

export default function ClientPage({ company, initialPosts }: { company: Company, initialPosts: Post[] }) {
    const router = useRouter();

    // Form State
    const [topText, setTopText] = useState("");
    const [subject, setSubject] = useState("");
    const [type, setType] = useState("POST");
    const [date, setDate] = useState(new Date().toISOString().split('T')[0]);
    const [editingPostId, setEditingPostId] = useState<string | null>(null);
    const [isSaving, setIsSaving] = useState(false);

    // UI State
    const [showEditCompanyModal, setShowEditCompanyModal] = useState(false);
    const [showAIModal, setShowAIModal] = useState(false);
    const [toast, setToast] = useState<{ message: string; type: ToastType; isVisible: boolean }>({
        message: "",
        type: "success",
        isVisible: false,
    });

    const showToast = (message: string, type: ToastType = "success") => {
        setToast({ message, type, isVisible: true });
    };

    const finalOutput = `${topText}\n\n${company.template}`;

    const handleCopy = async (text: string) => {
        try {
            await navigator.clipboard.writeText(text);
            showToast("Kopyalandı!", "success");
        } catch (err) {
            console.error("Failed to copy!", err);
            showToast("Kopyalama başarısız", "error");
        }
    };

    const handleSaveAndCopy = async () => {
        if (!topText.trim()) return;
        setIsSaving(true);
        try {
            if (editingPostId) {
                await updatePost(editingPostId, topText, finalOutput, subject || undefined, type, new Date(date));
                showToast("İçerik güncellendi ve kopyalandı!");
                setEditingPostId(null);
            } else {
                await createPost(company.id, topText, finalOutput, subject || undefined, type, new Date(date));
                showToast("İçerik kaydedildi ve kopyalandı!");
            }

            await handleCopy(finalOutput);

            // Reset form
            setTopText("");
            setSubject("");
            // Keep date and type for convenience
        } catch (error) {
            console.error("Failed to save", error);
            showToast("Bir hata oluştu", "error");
        } finally {
            setIsSaving(false);
        }
    };

    const handleEditPost = (post: Post) => {
        setTopText(post.content);
        setSubject(post.subject || "");
        setType(post.type);
        // Format date for input type="date"
        if (post.scheduledDate) {
            setDate(new Date(post.scheduledDate).toISOString().split('T')[0]);
        }
        setEditingPostId(post.id);
        showToast("Düzenleme modu aktif", "info");
    };

    const handleCancelEdit = () => {
        setEditingPostId(null);
        setTopText("");
        setSubject("");
        showToast("Düzenleme iptal edildi", "info");
    };

    const handleDeletePost = async (id: string) => {
        if (!confirm("Silmek istediğine emin misin?")) return;
        try {
            await deletePost(id);
            showToast("İçerik silindi", "info");
            if (editingPostId === id) handleCancelEdit();
        } catch (error) {
            showToast("Silinemedi", "error");
        }
    };

    const handleDeleteCompany = async () => {
        if (!confirm(`"${company.name}" firmasını ve tüm içeriklerini silmek istediğine emin misin? Bu işlem geri alınamaz!`)) return;
        await deleteCompany(company.id);
        router.replace("/");
    };

    return (
        <div className="p-4 lg:p-8 max-w-5xl mx-auto h-full overflow-y-auto relative text-zinc-900 bg-white pt-16 lg:pt-8 custom-scrollbar">
            <Toast
                message={toast.message}
                type={toast.type}
                isVisible={toast.isVisible}
                onClose={() => setToast(prev => ({ ...prev, isVisible: false }))}
            />

            {showEditCompanyModal && (
                <EditCompanyModal
                    company={company}
                    onClose={() => setShowEditCompanyModal(false)}
                />
            )}

            {showAIModal && (
                <AIModal
                    onClose={() => setShowAIModal(false)}
                    onGenerate={(text) => {
                        setTopText(text);
                        // Extract first word as subject if simple, or let user decide
                        if (!subject && text.length > 0) {
                            // Try to guess a subject from prompt or result? Nah, keep it simple.
                        }
                        showToast("AI tarafından oluşturuldu!", "success");
                    }}
                />
            )}

            <header className="flex items-center justify-between mb-8 pb-4 border-b border-zinc-100">
                <div>
                    <h1 className="text-2xl font-bold flex items-center gap-2 text-zinc-900">
                        {company.name}
                        <button
                            onClick={() => setShowEditCompanyModal(true)}
                            className="text-zinc-400 hover:text-blue-500 transition-colors"
                            title="Firmayı Düzenle"
                        >
                            <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" /></svg>
                        </button>
                    </h1>
                    <p className="text-xs text-zinc-500 font-mono mt-1 opacity-70 truncate max-w-md">{company.template.substring(0, 100)}...</p>
                </div>
                <button
                    onClick={handleDeleteCompany}
                    className="text-xs text-red-600 hover:text-red-700 bg-red-50 hover:bg-red-100 px-3 py-1.5 rounded-md"
                >
                    Firmayı Sil
                </button>
            </header>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                <div className="space-y-4">
                    {/* Metadata Inputs */}
                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                        <div>
                            <label className="block text-xs font-medium text-zinc-500 mb-1">Tarih</label>
                            <input
                                type="date"
                                className="block w-full rounded-md border border-zinc-300 py-2 px-3 text-sm focus:border-blue-500 focus:ring-blue-500 bg-white"
                                value={date}
                                onChange={(e) => setDate(e.target.value)}
                            />
                        </div>
                        <div>
                            <label className="block text-xs font-medium text-zinc-500 mb-1">Konu</label>
                            <input
                                type="text"
                                className="block w-full rounded-md border border-zinc-300 py-2 px-3 text-sm focus:border-blue-500 focus:ring-blue-500 bg-white"
                                placeholder="Örn: İndirim"
                                value={subject}
                                onChange={(e) => setSubject(e.target.value)}
                            />
                        </div>
                    </div>

                    <div>
                        <label className="block text-xs font-medium text-zinc-500 mb-1">Tür</label>
                        <div className="flex rounded-md shadow-sm">
                            <button
                                type="button"
                                onClick={() => setType("POST")}
                                className={`flex-1 px-3 py-2 text-sm font-medium border border-r-0 rounded-l-md ${type === "POST"
                                    ? "bg-blue-600 text-white border-blue-600"
                                    : "bg-white text-zinc-700 border-zinc-200 hover:bg-zinc-50"
                                    }`}
                            >
                                Post
                            </button>
                            <button
                                type="button"
                                onClick={() => setType("REELS")}
                                className={`flex-1 px-3 py-2 text-sm font-medium border rounded-r-md ${type === "REELS"
                                    ? "bg-pink-600 text-white border-pink-600"
                                    : "bg-white text-zinc-700 border-zinc-200 hover:bg-zinc-50"
                                    }`}
                            >
                                Reels
                            </button>
                        </div>
                    </div>

                    <div className={`rounded-xl border transition-all p-1 shadow-sm focus-within:ring-2 focus-within:ring-blue-500 relative ${editingPostId ? "border-blue-400 ring-1 ring-blue-400 bg-blue-50/50" : "border-zinc-200 bg-white"}`}>
                        <button
                            onClick={() => setShowAIModal(true)}
                            className="absolute top-2 right-2 p-1.5 text-purple-600 hover:bg-purple-50 rounded-md transition-colors z-10"
                            title="AI ile Oluştur"
                        >
                            <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M13 10V3L4 14h7v7l9-11h-7z" /></svg>
                        </button>
                        <textarea
                            className="block w-full rounded-lg border-0 bg-transparent py-4 px-4 text-zinc-900 placeholder:text-zinc-400 focus:ring-0 sm:text-base sm:leading-6 min-h-[150px] resize-y"
                            placeholder={editingPostId ? "İçeriği düzenle..." : "İçeriği buraya yaz..."}
                            value={topText}
                            onChange={(e) => setTopText(e.target.value)}
                        />
                        <div className="bg-zinc-50 rounded-b-lg p-3 text-xs text-zinc-500 border-t border-zinc-100 font-mono whitespace-pre-wrap">
                            {company.template}
                        </div>
                    </div>

                    <div className="flex gap-2">
                        {editingPostId && (
                            <button
                                onClick={handleCancelEdit}
                                className="flex-1 bg-zinc-200 hover:bg-zinc-300 text-zinc-800 font-semibold py-3 px-4 rounded-lg transition-colors"
                            >
                                İptal
                            </button>
                        )}
                        <button
                            onClick={handleSaveAndCopy}
                            disabled={isSaving || !topText}
                            className={`flex-[2] flex items-center justify-center gap-2 rounded-lg px-4 py-3 text-sm font-semibold text-white shadow-sm transition-all disabled:opacity-50 disabled:cursor-not-allowed ${type === "REELS"
                                ? "bg-pink-600 hover:bg-pink-500 focus-visible:outline-pink-600"
                                : "bg-blue-600 hover:bg-blue-500 focus-visible:outline-blue-600"
                                }`}
                        >
                            {isSaving ? "İşleniyor..." : editingPostId ? "Güncelle & Kopyala" : "Kaydet & Kopyala"}
                        </button>
                    </div>
                </div>

                <div className="space-y-4 flex flex-col h-full">
                    <h3 className="font-semibold text-zinc-900">Geçmiş İçerikler</h3>
                    <div className="space-y-4 flex-1 overflow-y-auto pr-2 custom-scrollbar min-h-[300px]">
                        {initialPosts.length === 0 && (
                            <p className="text-zinc-500 text-sm py-10 text-center border-2 border-dashed border-zinc-200 rounded-lg">Henüz içerik girilmemiş.</p>
                        )}
                        {initialPosts.map(post => {
                            const isEditing = editingPostId === post.id;
                            return (
                                <div
                                    key={post.id}
                                    className={`group relative rounded-lg border p-4 shadow-sm transition-all ${isEditing
                                        ? "border-blue-500 bg-blue-50"
                                        : "border-zinc-200 bg-white hover:shadow-md"
                                        }`}
                                >
                                    <div className="flex items-center justify-between mb-2">
                                        <div className="flex items-center gap-2">
                                            <span className={`inline-flex items-center rounded-md px-2 py-1 text-xs font-medium ring-1 ring-inset ${post.type === "REELS"
                                                ? "bg-pink-50 text-pink-700 ring-pink-600/10"
                                                : "bg-blue-50 text-blue-700 ring-blue-700/10"
                                                }`}>
                                                {post.type}
                                            </span>
                                            {post.subject && (
                                                <span className="text-xs font-medium text-zinc-500 truncate max-w-[150px]">
                                                    {post.subject}
                                                </span>
                                            )}
                                        </div>
                                        <time className="text-xs text-zinc-400">
                                            {post.scheduledDate ? new Date(post.scheduledDate).toLocaleDateString("tr-TR") : "Tarihsiz"}
                                        </time>
                                    </div>

                                    <p className="text-sm text-zinc-800 whitespace-pre-wrap mb-3 leading-relaxed">
                                        {post.content}
                                    </p>

                                    <div className="flex items-center justify-between pt-2 border-t border-zinc-50">
                                        <div className="text-xs text-zinc-400 opacity-0 group-hover:opacity-100 transition-opacity">
                                            Oluşturma: {new Date(post.createdAt).toLocaleDateString("tr-TR")}
                                        </div>
                                        <div className="flex gap-3">
                                            <button
                                                onClick={() => handleCopy(post.fullText)}
                                                className="text-xs font-medium text-blue-600 hover:text-blue-500"
                                            >
                                                Kopyala
                                            </button>
                                            <button
                                                onClick={() => handleEditPost(post)}
                                                className="text-xs font-medium text-zinc-500 hover:text-zinc-700"
                                            >
                                                Düzenle
                                            </button>
                                            <button
                                                onClick={() => handleDeletePost(post.id)}
                                                className="text-xs font-medium text-zinc-400 hover:text-red-500 transition-colors"
                                            >
                                                Sil
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            );
                        })}
                    </div>
                </div>
            </div>
        </div>
    );
}
