"use client";

import { useState } from "react";
import { updateCompany } from "@/app/actions";
import type { Company } from "@prisma/client";

interface EditCompanyModalProps {
    company: Company;
    onClose: () => void;
}

export default function EditCompanyModal({ company, onClose }: EditCompanyModalProps) {
    const [name, setName] = useState(company.name);
    const [template, setTemplate] = useState(company.template);
    const [isLoading, setIsLoading] = useState(false);

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setIsLoading(true);
        try {
            await updateCompany(company.id, name, template);
            onClose();
        } catch (error) {
            console.error(error);
        } finally {
            setIsLoading(false);
        }
    };

    return (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/50 backdrop-blur-sm p-4">
            <div className="w-full max-w-md bg-white rounded-xl shadow-xl p-6 relative animate-in zoom-in-95 duration-200">
                <button
                    onClick={onClose}
                    className="absolute top-4 right-4 text-zinc-400 hover:text-zinc-600"
                >
                    ✕
                </button>
                <h2 className="text-xl font-bold mb-6 text-zinc-900">Firma Düzenle</h2>
                <form onSubmit={handleSubmit} className="space-y-4">
                    <div>
                        <label className="block text-sm font-medium text-zinc-700 mb-1">Firma Adı</label>
                        <input
                            type="text"
                            required
                            className="w-full rounded-md border-zinc-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 py-2 px-3 text-zinc-900"
                            value={name}
                            onChange={(e) => setName(e.target.value)}
                        />
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-zinc-700 mb-1">
                            Sabit Alt Metin (Şablon)
                            <span className="text-xs text-zinc-500 ml-2 font-normal">Her postun altına eklenecek</span>
                        </label>
                        <textarea
                            required
                            rows={4}
                            className="w-full rounded-md border-zinc-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 py-2 px-3 text-sm font-mono text-zinc-900"
                            value={template}
                            onChange={(e) => setTemplate(e.target.value)}
                        />
                    </div>
                    <button
                        type="submit"
                        disabled={isLoading}
                        className="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-4 rounded-md transition-colors disabled:opacity-50"
                    >
                        {isLoading ? "Güncelleniyor..." : "Güncelle"}
                    </button>
                </form>
            </div>
        </div>
    );
}
